/***************************************************************************
 *
 * Copyright (c) 2013 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <string>
#include <sstream>
#include <fstream>
#include <iostream>

#include "Log.h"

#include "CalibrationHelpers.h"
#include "Subdivision.h"

using namespace std;
using namespace LayerManagerCalibration;

bool Subdivision::parseStream(ifstream& stream)
{
    uint pos = stream.tellg();
    string line;

    while (getline(stream, line))
    {
        size_t delimiter = line.find_first_of(':');

        if (delimiter == string::npos)
        {
            LOG_WARNING("Subdivision",
                        "Line found without colon, line=" << line);
            return false;
        }

        string key = line.substr(0, delimiter);
        string value = line.substr(delimiter + 1);

        key = trim(key);
        value = trim(value);

        if (key.find_first_of('-') == 0)
        {
            // Reached beginning of a new subdivision
            stream.seekg(pos, std::ios_base::beg);
            return true;
        }

        if (key.compare("name") == 0)
        {
            m_sName = value;
            m_bNameParsed = true;
            LOG_INFO("Subdivision", "Found name=" << m_sName);
        }
        else if (key.compare("id") == 0)
        {
            istringstream iss(value);
            iss >> m_u32Id;
            if (iss.fail())
            {
                LOG_WARNING("Subdivision", "Found invalid ID=" << value);
                return false;
            }
            else
            {
                m_bIdParsed = true;
                LOG_INFO("Subdivision", "Found ID=" << m_u32Id);
            }
        }
        else if (key.compare("top left x") == 0)
        {
            istringstream iss(value);
            iss >> m_u32TopLeftCoordinateX;
            if (iss.fail())
            {
                LOG_WARNING("Subdivision",
                            "Found invalid top left x=" << value);
                return false;
            }
            else
            {
                m_bTopLeftXParsed = true;
                LOG_INFO("Subdivision",
                         "Found top left x=" << m_u32TopLeftCoordinateX);
            }
        }
        else if (key.compare("top left y") == 0)
        {
            istringstream iss(value);
            iss >> m_u32TopLeftCoordinateY;
            if (iss.fail())
            {
                LOG_WARNING("Subdivision",
                            "Found invalid top left y=" << value);
                return false;
            }
            else
            {
                m_bTopLeftYParsed = true;
                LOG_INFO("Subdivision",
                         "Found top left y=" << m_u32TopLeftCoordinateY);
            }
        }
        else if (key.compare("bottom right x") == 0)
        {
            istringstream iss(value);
            iss >> m_u32BottomRightCoordinateX;
            if (iss.fail())
            {
                LOG_WARNING("Subdivision",
                            "Found invalid bottom right x=" << value);
            }
            else
            {
                m_bBottomRightXParsed = true;
                LOG_INFO("Subdivision",
                         "Found bottom right x="
                         << m_u32BottomRightCoordinateX);
            }
        }
        else if (key.compare("bottom right y") == 0)
        {
            istringstream iss(value);
            iss >> m_u32BottomRightCoordinateY;
            if (iss.fail())
            {
                LOG_WARNING("Subdivision",
                            "Found invalid bottom right y=" << value);
            }
            else
            {
                m_bBottomRightYParsed = true;
                LOG_INFO("Subdivision",
                         "Found bottom right y="
                         << m_u32BottomRightCoordinateY);
            }
        }
        else if (key.compare("uncalibrated") == 0)
        {
            if (value.compare("true") == 0)
            {
                m_bUncalibrated = true;
                LOG_INFO("Subdivision", "Found uncalibrated=true");
            }
            else if (value.compare("false") == 0)
            {
                m_bUncalibrated = false;
                LOG_INFO("Subdivision", "Found uncalibrated=false");
            }
            else
            {
                LOG_WARNING("Subdivision",
                            "Found invalid uncalibrated, value=" << value);
            }
        }
        else
        {
            if (!handleUnknownKey(key, value))
            {
                LOG_WARNING("Subdivision",
                            "Unknown or invalid key in subdivision, "
                            "value=" << key);
                return false;
            }
        }
        pos = stream.tellg();
    }

    return true;
}

bool Subdivision::validate()
{
    bool valid = true;

    // Verify that name was defined
    if (!m_bNameParsed)
    {
        LOG_WARNING("Subdivision", "No name defined");
        valid = false;
    }

    // Verify that id was defined
    if (!m_bIdParsed)
    {
        LOG_WARNING("Subdivision", "No id defined");
        valid = false;
    }

    // Verify that the top left coordinate was defined
    if (!m_bTopLeftXParsed)
    {
        LOG_WARNING("Subdivision", "No top left x defined");
        valid = false;
    }
    if (!m_bTopLeftYParsed)
    {
        LOG_WARNING("Subdivision", "No top left y defined");
        valid = false;
    }

    // Verify that the bottom right coordinate was defined
    if (!m_bBottomRightXParsed)
    {
        LOG_WARNING("Subdivision", "No bottom right x defined");
        valid = false;
    }
    if (!m_bBottomRightYParsed)
    {
        LOG_WARNING("Subdivision", "No bottom right y defined");
        valid = false;
    }

    // Verify that the bottom right coordinate is in the bottom right
    if ((m_u32TopLeftCoordinateX >= m_u32BottomRightCoordinateX)
        || (m_u32TopLeftCoordinateY >= m_u32BottomRightCoordinateY))
    {
        LOG_WARNING("Subdivision",
                    "Subdivision area has zero or negative size");
        valid = false;
    }

    return valid;
}

void Subdivision::clipCoordinate(const coordinate& coordIn, coordinate& coordOut)
{
    if (coordIn.x < m_u32TopLeftCoordinateX)
    {
        coordOut.x = m_u32TopLeftCoordinateX;
    }
    else if (coordIn.x > m_u32BottomRightCoordinateX)
    {
        coordOut.x = m_u32BottomRightCoordinateX;
    }
    else
    {
        coordOut.x = coordIn.x;
    }

    if (coordIn.y < m_u32TopLeftCoordinateY)
    {
        coordOut.y = m_u32TopLeftCoordinateY;
    }
    else if (coordIn.y > m_u32BottomRightCoordinateY)
    {
        coordOut.y = m_u32BottomRightCoordinateY;
    }
    else
    {
        coordOut.y = coordIn.y;
    }
}
